#!/usr/bin/bash

set -e

usage() {
cat << HELP_END
A wrapper script to run scopes.py, to produce 'full' scope_info file
from ibnetdiscover output. Can process prepared ibnetdiscover output or
try to run ibnetdiscover itself

$0 [-C] [-f <scope_info>] [-d <distances_file>] [-o order_info] [-a]
     (-i <ibnetdiscover_cli> | <ibnetdiscover_output>)

-C -- do not remove cache directory
-f <scope_info> -- output filename for 'full' scope_info file. Use - for stdout.
-d <distances> -- calculate distances between all pairs of TORs and write them
                    to file <distances>.
-t <topofile> -- extract topology information and write to file <topofile>
-o <order_info> -- create order_info file for testing bisectional bandwidth
                     by distances between TORs.
-a -- Write all possible pairs to order_info file.
-n -- Only ports connected to nodes are counted as downlinks when calculating
      oversubscribe ratio (default is to count all ports except connected to
      switches and special nodes).
-i <ibnetdiscover_cli> -- run ibnetdiscover and pass <ibnetdiscover_switches>
                            to it. Several switches can be used if quoted like
                            $0 -i "-C mlx5_0 -P 1"
                            Note that ibnetdiscover may require additional
                            permissions to run.
<ibnetdiscover_output> -- filename of ibnetdiscover output. Use - for stdin.
                            Mutually exclusive with -i switch.
HELP_END
}

source_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
VENV_DIR=venv

scope_info=""
distances=""
order_info=""
topofile=""
downlink_nodes_only=false
all_pairs=false
run_ibnetdiscover=false
ibnetdiscover_cli=""
ibnetdiscover_output=""
scopes_py_cli=""
clean=true

while getopts ":f:d:t:o:aiCn" o; do
    case "${o}" in
        C)
            clean=false
            ;;
        f)
            scope_info="${OPTARG}"
            scopes_py_cli+=" -f ${scope_info}"
            ;;
        d)
            distances="${OPTARG}"
            scopes_py_cli+=" -d ${distances}"
            ;;
        t)
            topofile="${OPTARG}"
            scopes_py_cli+=" -t ${topofile}"
            ;;
        n)
            downlink_nodes_only=true
            scopes_py_cli+=" --downlink-nodes-only"
            ;;
        o)
            order_info="${OPTARG}"
            scopes_py_cli+=" -o ${order_info}"
            ;;
        a)
            all_pairs=true
            scopes_py_cli+=" -a"
            ;;
        i)
            run_ibnetdiscover=true
            nextarg="${!OPTIND}"
            if [[ -n "${nextarg}" ]]; then
                ibnetdiscover_cli="${nextarg}"
                OPTIND=$((OPTIND + 1))
            fi
            scopes_py_cli+=" -"
            ;;
        :)
            usage
            exit 1
            ;;
        *)
            usage
            exit 1
            ;;
    esac
done

if [[ ${OPTIND} -eq $# ]]; then
    # There's one positional argument
    if [[ "${run_ibnetdiscover}" = true ]]; then
        # -d is mutually exclusive with giving ibnetdiscover output filename
        usage
        exit 1
    fi
    ibnetdiscover_output="${!OPTIND}"
    scopes_py_cli+=" ${ibnetdiscover_output}"
elif [[ $((OPTIND - 1)) -eq $# ]]; then
    if [[ "${run_ibnetdiscover}" = false ]]; then
        # Either -d or ibnetdiscover output filename should be given
        usage
        exit 1
    fi
fi

python="python"
version=$( python -V 2>&1 )
if [[ "$version" == *" 2."* ]];then
    python+="3"
fi

run_venv() {
    source $VENV_DIR/bin/activate
}

set_venv() {
    echo "Setting virtual environment..."
    $python -m venv $VENV_DIR
    if [[ $? -ne 0 ]]; then
        echo "Error - could not set virtual environment"
        return 1
    fi
    run_venv
    $python -m pip install --upgrade pip --no-cache-dir > /dev/null
    $python -m pip install -r $source_dir/requirements.txt --no-cache-dir > /dev/null
}

do_start() {
    local res=0
    if [[ ! -d $VENV_DIR ]]; then
        set_venv
        res=$?
    else
        run_venv
    fi
    return $res
}

do_end() {
    deactivate
    if [[ "${clean}" = true ]]; then
        echo "Cleaning cache directory"
        rm -rf $VENV_DIR
    fi
}

do_start || exit 1

if [[ "${run_ibnetdiscover}" == true ]]; then
    ibnetdiscover ${ibnetdiscover_cli} | ${source_dir}/scopes.py ${scopes_py_cli}
else
    ${source_dir}/scopes.py ${scopes_py_cli}
fi

do_end
