#!/usr/bin/bash
source_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
INI_FILE_NAME=output_config.ini
outputdir=
normalize=false
report=false
clean=false
per_hca=true
OUTPUT_CACHE_DIR=$source_dir/ck_output_cache
VENV_DIR=$OUTPUT_CACHE_DIR/venv
IBDIAGNET_OUTPUT_DIR=$OUTPUT_CACHE_DIR/ibdiagnet
NORMALIZE_ERR_FILE=$OUTPUT_CACHE_DIR/normalize_latency.err
ibdiagnet_output_dir=$IBDIAGNET_OUTPUT_DIR
IBDIAGNET_ARGS="-r --routers --get_cable_info --enable_output all -o $IBDIAGNET_OUTPUT_DIR "
IBDIAGNET_EXE="${IBDIAGNET_EXE:-`which ibdiagnet`}"
GEN_OUTPUT_SCRIPT=$source_dir/generate_output.py
OUTPUT_INI=$source_dir/$INI_FILE_NAME

usage() {
cat <<HELP_USAGE

A wrapper script for output files processing. Generates heatmaps of bandwidth, latency and normalized latency,
puts all heatmaps in a zip file and creates a tgz of JSON files in output directory.
NOTE - set IBDIAGNET_EXE for location of ibdiagnet executable, otherwise will use default path.

$0 -o|--outputdir <dir> [Optional Parameters]

	-o|--outputdir <dir>        Path to clusterkit output directory

  Optional Parameters:
	-l|--normalize              Normalize latency.json. Requires either an existing ibdiagnet data_dir field
	                            (see INI file), or an execution of ibdiagnet within the script,
	                            which requires sudo permission                             default: false
	-N|--per_node               Clusterkit output is per node, not per HCA.                default: false
	-C|--clean                  Remove cache directory                                     default: false
	-R|--report                 create report (PDF)                                        default: false


  Configure execution using INI file $OUTPUT_INI:

  [DEFAULT] section:
      verbose            Boolean, set to 1/0 for more/less information while running, respectively


  [ibdiagnet] section:
      ** Relevant when running script with --normalize
      hca_device         HCA to use when running ibdiagnet on the fabric
      data_dir           ibdiagnet output directory made with the next command
                         "ibdiagnet -r --routers --get_cable_info --enable_output all"
                         the following files are required:
                	      [ibdiagnet2.db_csv, ibdiagnet2.fdbs, ibdiagnet2.cables, ibdiagnet2.nodes_info]

  [normalize_latency] section:
      config_file	 CSV file of devices and optical cables. Each line can be either a device ID with
        	         worst case latency in ns or an optical cable part number.
	                 For example see latency_calc_config.csv

  [render_heatmap] section:
      *_thresholds       Comma separated list of threshold values for heatmap color scale.
        	         May contain 1/2/3/6 values to set 2/3/4/7 color scale bar, respectively.
                         The values can contain the following variables:
                           ['average','Average','avg','mean','std','STD','sigma'].
                         Examples (in INI file, in render_heatmap section):
                              latency_thresholds=0.4,0.5,0.6,0.7,0.8,0.9
                              # 7 colors scale for latency in usec

                              bandwidth_thresholds=avg-std,avg+std,avg+2*std
                              # 4 colors scale with jumps of standard deviation around the mean for bandwidth heatmap

                              latency_thresholds=mean-0.1,mean+0.1
                              # 3 colors scale for latencies under mean-0.1 usec, between mean-0.1 to mean+0.1
                              # and above mean+0.1

                              bandwidth_thresholds=mean
                              # 2 colors scale for below and above the mean of bandwidth
HELP_USAGE
}

run_venv() {
	source $VENV_DIR/bin/activate
}

set_venv() {
	echo "Setting virtual environment..."
	$python -m venv $VENV_DIR
	if [[ $? -ne 0 ]]; then
		echo "Error - could not set virtual environment"
		return 1
	fi
	run_venv
	python -m pip install --upgrade pip --no-cache-dir > /dev/null
	python -m pip install -r $source_dir/requirements.txt --no-cache-dir > /dev/null
}

do_start() {
	local res=0
 	if [[ ! -d $VENV_DIR ]]; then
		set_venv
		res=$?
	else
		run_venv
	fi
 	return $res
}

do_end() {
	deactivate
	if [[ "$clean" = true ]]; then
		echo "Cleaning cache directory"
		rm -rf $VENV_DIR
		rm -rf $IBDIAGNET_OUTPUT_DIR
		rm -rf $OUTPUT_CACHE_DIR
	fi
}

render() {
	json_file=$1
	config_file=$2
	if [[ $json_file && $config_file ]]; then
		echo "Rendering  ${json_file}"
	 	python $GEN_OUTPUT_SCRIPT -r -f $config_file -j $json_file 
	fi
}

render_output_files() {
	files="latency.json bandwidth.json"
	for file in $files; do
		file_path="$outputdir/$file"
		file_path=$( realpath $file_path 2> /dev/null )
		if [[ -f $file_path ]]; then
			render $file_path $OUTPUT_INI
		fi
	done
}

check_ibd_files() {
	local code=0
	local data_dir="${1}"
	files="ibdiagnet2.db_csv ibdiagnet2.fdbs ibdiagnet2.cables ibdiagnet2.nodes_info"
	for f in $files;do
		if [[ ! -f "${data_dir}/$f" ]]; then
			echo "ERROR: missing file $f in ${data_dir}"
			code=1
		fi
	done
	return $code
}

check_ibd_dir() {
	local data_dir="${1}"
	if [[ ! -d "${data_dir}" ]]; then
		echo "ERROR: ${data_dir} directory does not exist/not a directory"
		return 1
	fi
	check_ibd_files "${data_dir}"
	return $?
}

run_ibdiagnet_all_devices() {
	devices=$( ibstat -l )
	cables_file="${IBDIAGNET_OUTPUT_DIR}/ibdiagnet2.cables"
	for d in $devices; do
		sudo $IBDIAGNET_EXE $IBDIAGNET_ARGS -i $d >/dev/null
		if [[ -f "$cables_file" ]]; then
			return 0
		fi
		rm -rf $IBDIAGNET_OUTPUT_DIR
	done
	return 1
}

run_ibdiagnet() {
	if [ -z "$IBDIAGNET_EXE" ]; then
		echo "IBDIAGNET_EXE was not set and no location of ibdiagnet was found."
		return 1
	fi

	echo -n "Checking sudo permission for ibdiagnet... "
	out=$( timeout --kill-after=1 1 sudo whoami )
	if [ "$out" = root ];then
		echo "yes"
	else
		echo "no"
		return 1
	fi

	hca_arg=$( awk -F "=" '/^hca_device/ {print $2}' $OUTPUT_INI )
	if [[ $hca_arg ]]; then
		sudo $IBDIAGNET_EXE $IBDIAGNET_ARGS -i $hca_arg >/dev/null
	else
		run_ibdiagnet_all_devices
	fi
	return $?
}

ibdiagnet_dir_is_set() {
	if [[ ! -d $IBDIAGNET_OUTPUT_DIR ]]; then
		data_dir=$( awk -F "=" '/^data_dir/ {print $2}' $OUTPUT_INI )
		if [[ -n "${data_dir}" ]]; then
			check_ibd_dir "${data_dir}"
			if [[ $? == 0 ]]; then
				ibdiagnet_output_dir="${data_dir}"
				return 0
			fi
		fi
		return 1
	fi
	check_ibd_files $IBDIAGNET_OUTPUT_DIR
	return $?
}

generate_pdf_report(){
echo "Creating a PDF report..."
	local cmd="python $GEN_OUTPUT_SCRIPT -p -d $outputdir"
	eval "$cmd"
	exit_status=$?
	if [ $exit_status -ne 0 ]; then
		echo
		echo "ERROR - can not create a PDF report"
	fi
}

normalize_latency_and_render() {
	ibdiagnet_dir_is_set || run_ibdiagnet
	if [[ $? != 0 ]]; then
		echo "Cannot normalize latency"
		return
	fi

	latencyfile=$( realpath $outputdir/latency.json  2> /dev/null )
	if [[ ! -f $latencyfile ]]; then
		echo "No latency file found in directory - cannot normalize"
		return
	fi
	echo "Normalizing ${latencyfile}"
	local cmd="python $GEN_OUTPUT_SCRIPT -n -f $OUTPUT_INI -j $latencyfile -d $ibdiagnet_output_dir"
	if [[ "$per_hca" = true ]]; then
		cmd+=" -p"
	fi
	eval "$cmd" 2> $NORMALIZE_ERR_FILE
	exit_status=$?
	if [ $exit_status -ne 0 ]; then
		echo
		echo "An error occurred while normalizing $( basename ${latencyfile}), cannot render."
		echo "This usually happens when cache directory contains files of another cluster."
		echo "Please erase cache directory (${OUTPUT_CACHE_DIR}) and rerun"
		echo "Full error is in ${NORMALIZE_ERR_FILE}"
		return
	fi

	normalized_file=$( realpath $outputdir/latency_normalized.json 2> /dev/null )
	if [[ ! -f $normalized_file ]]; then
		echo "Cannot create normalized latency heatmap - No json made"
		return
	fi
	render $normalized_file $OUTPUT_INI	
}

compress_output_files() {
	# zip all heatmap images
	cd $outputdir
	pngs=$( find -name "*.png" )
	if [ -z "$pngs" ]; then
		echo "Did not find any heatmaps in $outputdir"
	else
		echo Creating zip file..
		zip processed_results.zip $pngs
	fi

	# tgz all json files
	jsons=$( find -name "*.json" )
	if [ -z "$jsons" ]; then
		echo "Did not find any JSON file in $outputdir"
	else
		echo Creating tgz file...
		name=$( basename $outputdir )
		tar -czvf "${name}.tgz" $jsons
	fi
	cd - &> /dev/null
}

while [[ $# -gt 0 ]]
do
	key="${1}"
	case ${key} in
		-o|--outputdir)
			outputdir="${2}"
			shift # past argument
			;;
		-l|--normalize)
			normalize=true
			;;
		-R|--report)
			report=true
			;;	
		-N|--per_node)
			per_hca=false
			;;
		-C|--clean)
			clean=true
			;;
		-h|--help)
			usage
			exit 0
			;;
		*)    # unknown option
			echo Error: unknown option $key
			usage
			exit 1
			;;
	esac
    shift
done

if [ -z $outputdir ]; then
	usage
	exit 1
fi
if [ ! -d $outputdir ];then
  echo "Error: ${outputdir} directory not exists/ is not a directory"
  exit 1
fi

python="python"
version=$( python -V 2>&1 )
if [[ "$version" == *" 2."* ]];then
	python+="3"
fi

do_start || exit 1
render_output_files
if [[ "$normalize" = true ]]; then
  normalize_latency_and_render
fi
if [[ "$report" = true ]]; then
  generate_pdf_report
fi
compress_output_files
do_end
