/*
 * Copyright (c) 2012-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

#ifndef TREE_MANAGER_H_
#define TREE_MANAGER_H_

#include <memory>
#include <queue>
#include <string>
#include <unordered_map>
#include <unordered_set>

#include "agg_types.h"
#include "option_manager.h"
#include "port_data.h"
#include "sub_tree_info.h"
#include "sub_tree_score.h"

class Fabric;
class AggNode;
class AggNodeFabricInfo;
class AggTree;
struct PathScore;
struct AnScore;
struct Permutation;

typedef std::vector<AnScore> VecAnScore;
typedef std::vector<PathScore> VecPathScore;
typedef std::unordered_set<AggNodeFabricInfo*> SetANInfo;
typedef std::set<TreeNode*> SetTreeNodePtr;

class TreeManager
{
    SetAnFabricInfoPtr m_dfp_leafs_;   // Used only for DFP topology, if static allocation is used

    ListTreeIds m_available_mc_tree_ids_;   // List of available multicast tree ids

   public:
    TreeManager() {}

    int BuildTrees(bool seamless_restart);
    int UpdateTrees();

    int BuildSatTrees();

    int DumpTrees() const;
    int DumpTreesState() const;

    sharp_trees_t AllocateMulticastTreeId();
    void FreeMulticastTreeId(sharp_trees_t tree_id);

   private:
    int ParseFabricTreesFile(FILE* f);

    // parse the node desc section of trees file
    AggNodeFabricInfo* ParseAggNodeInfo(const char* line, int line_num);

    // set Tree root: the tree node with no parent.
    // validate tree structure: one and only one root
    int SetTreesRoot();
    int SetTreesNodesRank();

    int ParseFabricTreesFile(bool is_seamless_restart);
    int CalculateTrees();

    int CalculateTreeTrees();
    int UpdateTreeTrees();

    // BFS Trees calculation functions
    int CreateBfsTree(AggTree* p_agg_tree, AggNodeFabricInfo* p_agg_node);
    int CalculateBfsTrees();

    // Hyper-Cube trees calculation functions
    int CalculateHyperCubeTrees();
    static int CreateHyperCubeKruskalTree(AggTree* p_agg_tree,
                                          AggNodeFabricInfo* p_root_agg_node,
                                          VecPathScore& path_score_vector,
                                          Permutation perm);
    static void KruskalTreeUnifySets(AggTree* p_agg_tree,
                                     AggNodeFabricInfo* p_agg_node1,
                                     AggNodeFabricInfo* p_agg_node2,
                                     std::vector<int>& an_set_vector,
                                     std::vector<std::set<int>>& set_id_set_vector);
    static bool KruskalTreeCheckSets(AggTree* p_agg_tree,
                                     AggNodeFabricInfo* p_agg_node1,
                                     AggNodeFabricInfo* p_agg_node2,
                                     std::vector<int>& an_set_vector,
                                     std::vector<std::set<int>>& set_id_set_vector);

    // Dragonfly Plus Trees calculation functions
    int CalculateDfpTrees();
    int UpdateDfpTrees();
    void SetDfpTreeHeightAndLeafsNumber();
    void SortDfpRoots(vector<uint8_t>& groups_root_load);

    void ClearDescendantLeafs();
    void BuildDescendantLeafs();
    uint16_t GetMaxDescendants(bool& is_partial_spanning);

    static int ParseNodeInfo(const char* line, int line_num, std::string& node_desc, port_key_t& port_guid);
    static int ParseHcaNodeInfo(const char* line, int line_num, std::string& node_desc, hca_port_key_t& port_guid);

    void SortTreeRootsByGroup();
    void UpdateDfpLeafs();
    const SetAnFabricInfoPtr& GetDfpLeafs() { return m_dfp_leafs_; }
};

#endif   // TREE_MANAGER_H_
