/*
 * Copyright (c) 2024-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * This software is available to you under the terms of the
 * OpenIB.org BSD license included below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

#pragma once

#include <string>
#include <ibis/ibis.h>
#include <infiniband/ibdiag/ibdiag_types.h>


//
// Section NODES
//  - SMP_NodeInfo
//
class NodeRecord {
    public:
        std::string             m_description;
        struct SMP_NodeInfo     m_data;

    public:
        NodeRecord() : m_data({}) {}

        static int Init(vector < ParseFieldInfo <class NodeRecord> > &parse_section_info);
};

//
// Section PORTS
//  - SMP_PortInfo
//
class PortRecord {
    public:
        u_int64_t               m_node_guid;
        u_int64_t               m_port_guid;
        u_int8_t                m_port_num;

        std::string             m_fec_actv;
        std::string             m_retrans_actv;
        bool                    m_has_na;

        struct SMP_PortInfo     m_data;

    public:
        PortRecord() : m_node_guid(0), m_port_guid(0), m_port_num(0), m_has_na(false), m_data({}) {}

        static int Init(vector < ParseFieldInfo <class PortRecord> > &parse_section_info);
};

//
// Section EXTENDED_NODE_INFO
//  - ib_extended_node_info
//
class ExtendedNodeInfoRecord {
    public:
        u_int64_t                        m_node_guid;
        struct ib_extended_node_info     m_data;
        bool                             m_has_na;

    public:
        ExtendedNodeInfoRecord() : m_node_guid(0), m_data({}), m_has_na(false) {}

        static int Init(vector < ParseFieldInfo <class ExtendedNodeInfoRecord> > &parse_section_info);
};

//
// Section EXTENDED_PORT_INFO
//  - SMP_MlnxExtPortInfo
//
class ExtendedPortInfoRecord {
    public:
        u_int64_t                     m_node_guid;
        u_int64_t                     m_port_guid;
        u_int8_t                      m_port_num;
        bool                          m_has_na;
        struct SMP_MlnxExtPortInfo    m_data;

    public:
        ExtendedPortInfoRecord() : m_node_guid(0), m_port_guid(0), m_port_num(0), m_has_na(false), m_data({}) {}

        static int Init(vector < ParseFieldInfo <class ExtendedPortInfoRecord> > &parse_section_info);
};

//
// Section PORT_INFO_EXTENDED
//  - SMP_PortInfoExtended
//
class PortInfoExtendedRecord {
    public:
        u_int64_t                   m_node_guid;
        u_int64_t                   m_port_guid;
        u_int8_t                    m_port_num;
        struct SMP_PortInfoExtended m_data;

    public:
        PortInfoExtendedRecord() : m_node_guid(0), m_port_guid(0), m_port_num(0), m_data({}) {}

        static int Init(vector < ParseFieldInfo <class PortInfoExtendedRecord> > &parse_section_info);
};

//
// Section SWITCHES
//  - SMP_SwitchInfo
//
class SwitchRecord {
    public:
        u_int64_t                 m_node_guid;
        struct SMP_SwitchInfo     m_data;

    public:
        SwitchRecord() : m_node_guid(0), m_data({}) {}

        static int Init(vector < ParseFieldInfo <class SwitchRecord> > &parse_section_info);
};

//
// Section PORT_HIERARCHY_INFO
//
class PortHierarchyInfoRecord
{
    public:
        u_int64_t m_node_guid;
        u_int64_t m_port_guid;
        u_int64_t m_template_guid;
        u_int8_t  m_port_num;

    public:
        int32_t  m_bus;
        int32_t  m_device;
        int32_t  m_function;
        int32_t  m_type;
        int32_t  m_slot_type;
        int32_t  m_slot_value;
        int32_t  m_asic;
        int32_t  m_cage;
        int32_t  m_port;
        int32_t  m_split;
        int32_t  m_ibport;
        int32_t  m_port_type;
        int32_t  m_asic_name;
        int32_t  m_is_cage_manager;
        int32_t  m_number_on_base_board;

        // DeviceRecord
        int32_t  m_device_num_on_cpu_node;
        int32_t  m_cpu_node_number;

        // BoardRecord
        int32_t m_board_type;
        int32_t m_chassis_slot_index;
        int32_t m_tray_index;

        int32_t m_topology_id;


        // APort
        int32_t  m_aport;
        int32_t  m_plane;
        int32_t  m_num_of_planes;

        bool     m_has_na;

    public:
        PortHierarchyInfoRecord()
            : m_node_guid(0), m_port_guid(0), m_template_guid(0),
              m_port_num(0), m_bus(0), m_device(0), m_function(0),
              m_type(0), m_slot_type(0), m_slot_value(0),
              m_asic(0), m_cage(0), m_port(0), m_split(0),
              m_ibport(0), m_port_type(0), m_asic_name(0),
              m_is_cage_manager(0), m_number_on_base_board(0),
              m_device_num_on_cpu_node(0),
              m_cpu_node_number(0),
              m_board_type(0),
              m_chassis_slot_index(0),
              m_tray_index(0),
              m_topology_id(0),
              m_aport(0), m_plane(0), m_num_of_planes(0), m_has_na(false)

        {}

        static int Init(vector < ParseFieldInfo <class PortHierarchyInfoRecord> > &parse_section_info);
};

//
// Section PHYSICAL_HIERARCHY_INFO
//
class PhysicalHierarchyInfoRecord
{
    public:
        u_int64_t m_node_guid;

    public:
        int32_t  m_campus_serial_num;
        int32_t  m_room_serial_num;
        int32_t  m_rack_serial_num;
        int32_t  m_system_type;
        int32_t  m_system_topu_num;
        int32_t  m_board_type;
        int32_t  m_board_slot_num;
        int32_t  m_device_serial_num;
        // template 6
        int32_t m_device_num_on_cpu_node;
        int32_t m_cpu_node_number;
        int32_t m_chassis_slot_index;
        int32_t m_tray_index;
        int32_t m_topology_id;
        bool     m_has_na;

    public:
        PhysicalHierarchyInfoRecord()
            :m_node_guid(0), m_campus_serial_num(0), m_room_serial_num(0),
             m_rack_serial_num(0), m_system_type(0), m_system_topu_num(0),
             m_board_type(0), m_board_slot_num(0), m_device_serial_num(0),
             m_device_num_on_cpu_node(0), m_cpu_node_number(0), m_chassis_slot_index(0),
             m_tray_index(0), m_topology_id(0), m_has_na(false)
        {}

    public:
        static int Init(vector < ParseFieldInfo <class PhysicalHierarchyInfoRecord> > &parse_section_info);
};

//
// Section AR_INFO
// - adaptive_routing_info
//
class ARInfoRecord {
    public:
        u_int64_t               m_node_guid;
        adaptive_routing_info   m_data;

    public:
        ARInfoRecord() : m_node_guid(0), m_data({}) {}

        static int Init(vector < ParseFieldInfo <class ARInfoRecord> > &parse_section_info);
};

//
// Section LINKS
//
class LinkRecord {
    public:
        u_int64_t   node_guid1;
        u_int8_t    port_num1;
        u_int64_t   node_guid2;
        u_int8_t    port_num2;

    public:
        LinkRecord(): node_guid1(0), port_num1(0), node_guid2(0), port_num2(0) {};

        static int Init(vector < ParseFieldInfo <class LinkRecord> > &parse_section_info);
};

//
// Section START_GENERAL_INFO_SMP
//
class GeneralInfoSMPRecord {
    public:
        u_int64_t     m_node_guid;
        std::string   m_fw_info_extended_major;
        std::string   m_fw_info_extended_minor;
        std::string   m_fw_info_extended_sub_minor;
        string        m_capability_mask_fields[NUM_CAPABILITY_FIELDS];

    public:
        GeneralInfoSMPRecord() : m_node_guid(0) {}

        static int Init(vector < ParseFieldInfo <class GeneralInfoSMPRecord> > &parse_section_info);
};

//
// Section START_EXTENDED_SWITCH_INFO
//  - SMP_ExtendedSwitchInfo
//
class ExtendedSwitchInfoRecord {
    public:
        u_int64_t                         m_node_guid;
        struct SMP_ExtendedSwitchInfo     m_data;

    public:
        ExtendedSwitchInfoRecord() : m_node_guid(0), m_data({}) {}

        static int Init(vector < ParseFieldInfo <class ExtendedSwitchInfoRecord> > &parse_section_info);
};

//
// Section NODES_INFO
//  - VendorSpec_GeneralInfo
//
class GeneralInfoGMPRecord {
    public:
        u_int64_t                     m_node_guid;
        struct VendorSpec_GeneralInfo m_data;
        bool                          m_has_na;
        bool                          m_cap_has_na;
        GeneralInfoGMPRecord() : m_node_guid(0), m_data({}), m_has_na(false), m_cap_has_na(false) {}

        static int Init(vector < ParseFieldInfo <class GeneralInfoGMPRecord> > &parse_section_info);
};

//
// Section PM_INFO
//  - PM_PortCounters
//  - PM_PortCountersExtended
//  - PM_PortRcvErrorDetails
//  - PM_PortXmitDiscardDetails
//
class PMInfoRecord {
    public:
        u_int64_t                           m_node_guid;
        u_int64_t                           m_port_guid;
        u_int8_t                            m_port_num;
        struct PM_PortCounters              m_pm_port_counters;
        struct PM_PortCountersExtended      m_pm_port_counters_ext;
        struct PM_PortRcvErrorDetails       m_pm_port_rcv_error_details;
        struct PM_PortXmitDiscardDetails    m_pm_port_xmit_discard_details;
    
        PMInfoRecord()
            : m_node_guid(0), m_port_guid(0), m_port_num(0),
              m_pm_port_counters({}), m_pm_port_counters_ext({}),
              m_pm_port_rcv_error_details({}), m_pm_port_xmit_discard_details({})
        {}
    
        static int Init(vector < ParseFieldInfo <class PMInfoRecord> > &parse_section_info);
};

//
// Section HBF_PORT_COUNTERS
//  - port_routing_decision_counters
//
class HBFPortCountersRecord {
    public:
        u_int64_t                               m_node_guid;
        u_int64_t                               m_port_guid;
        u_int8_t                                m_port_num;
        struct port_routing_decision_counters   m_data;
    
        HBFPortCountersRecord()
            : m_node_guid(0), m_port_guid(0), m_port_num(0), m_data({})
        {}
    
        static int Init(vector < ParseFieldInfo <class HBFPortCountersRecord> > &parse_section_info);
};

//
// Section FAST_RECOVERY_COUNTERS
//  - VS_FastRecoveryCounters
//
class FastRecoveryCountersRecord {
    public:
        u_int64_t                           m_node_guid;
        u_int64_t                           m_port_guid;
        u_int8_t                            m_port_num;
        struct VS_FastRecoveryCounters      m_data;
    
        FastRecoveryCountersRecord()
            : m_node_guid(0), m_port_guid(0), m_port_num(0), m_data({})
        {}
    
        static int Init(vector < ParseFieldInfo <class FastRecoveryCountersRecord> > &parse_section_info);
};

//
// Section CREDIT_WATCHDOG_TIMEOUT_COUNTERS
//  - VS_CreditWatchdogTimeoutCounters
//
class CreditWatchdogTimeoutCountersRecord {
    public:
        u_int64_t                               m_node_guid;
        u_int64_t                               m_port_guid;
        u_int8_t                                m_port_num;
        struct VS_CreditWatchdogTimeoutCounters m_data;
    
        CreditWatchdogTimeoutCountersRecord()
            : m_node_guid(0), m_port_guid(0), m_port_num(0), m_data({})
        {}
    
        static int Init(vector < ParseFieldInfo <class CreditWatchdogTimeoutCountersRecord> > &parse_section_info);
};

//
// Section RN_COUNTERS
//  - port_rn_counters
//
class RNCountersRecord {
    public:
        u_int64_t                           m_node_guid;
        u_int64_t                           m_port_guid;
        u_int8_t                            m_port_num;
        struct port_rn_counters             m_data;

        RNCountersRecord()
            : m_node_guid(0), m_port_guid(0), m_port_num(0), m_data({})
        {}

        static int Init(vector < ParseFieldInfo <class RNCountersRecord> > &parse_section_info);
};

//
// Section PM_PORT_SAMPLES_CONTROL
//  - PM_PortSamplesControl
//
class PMPortSamplesControlRecord {
 public:
        u_int64_t                               m_node_guid;
        u_int64_t                               m_port_guid;
        u_int8_t                                m_port_num;
        struct PM_PortSamplesControl            m_data;

        PMPortSamplesControlRecord()
            : m_node_guid(0), m_port_guid(0), m_port_num(0), m_data({})
        {}

        static int Init(vector < ParseFieldInfo <class PMPortSamplesControlRecord> > &parse_section_info);
};

//
// Section CHASSIS_INFO
//  - SMP_ChassisInfo
//
class ChassisInfoRecord {
 public:
        u_int64_t                               m_node_guid;
        struct SMP_ChassisInfo                  m_data;

        ChassisInfoRecord()
            : m_node_guid(0), m_data({})
        {}

        static int Init(vector < ParseFieldInfo <class ChassisInfoRecord> > &parse_section_info);
};
