# Copyright (C) Jan 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --


#######################################################
#
# DataPrinter.py
# Python implementation of the Class DataPrinter
# Generated by Enterprise Architect
# Created on:      19-Dec-2019 3:18:37 PM
# Original author: talve
#
#######################################################
import os
from resourceparse_lib.utils import constants as cs


class DataPrinter:
    """This class is responsible for set and manage the parser output.
    """
    def __init__(self, verbosity, out_file, out_dir):
        self._verbosity = verbosity
        self._out_file = out_file
        self._out_dir = out_dir
        self._top_notice_db = []

    def print_notice_before_parse(self, notice_msg):
        """This method prints notice message according the output type.
        """
        if self._verbosity > 0:
            if self._out_file:
                self._top_notice_db.append(notice_msg)
            else:
                print(notice_msg)

    def print_parsed_segment(self, parsed_segment_db, title, segment_separator):
        """This method prints the parsed segments after check if we need to print to a file or to screen.
        """
        if self._out_file:
            self._print_to_file(parsed_segment_db, title, segment_separator, self._out_file)
            print("write to file: ", self._out_file)
        elif self._out_dir:
            self._print_to_multiple_files(parsed_segment_db)
        else:
            self._print_to_screen(parsed_segment_db, title, segment_separator)

    def _print_to_screen(self, parsed_segment_db, title, segment_separator):
        """This method prints the parsed segments to the screen.
        """
        if title:
            print(title)
        for seg in parsed_segment_db:
            if segment_separator:
                print(segment_separator)
            parsed_seg = seg.get_parsed_data()
            for field in parsed_seg:
                print(field)
        if segment_separator:
            print(segment_separator)

    def _print_to_multiple_files(self, parsed_segment_db):
        """This method prints the parsed segments to multiple files, each containing a single segment.
        """
        os.makedirs(self._out_dir, exist_ok=True)
        # Count segment duplication in order to assign unique names later if necessary
        total_segment_occurence = {}
        for parsed_segment in parsed_segment_db:
            total_segment_occurence[parsed_segment.get_type()] = total_segment_occurence.get(parsed_segment.get_type(), 0) + 1
        current_segment_occurence = {}
        files = []
        for parsed_segment in parsed_segment_db:
            # Aggregate segment occurence and use the value only if the segment has duplicates
            occurence = current_segment_occurence[parsed_segment.get_type()] = current_segment_occurence.get(parsed_segment.get_type(), 0) + 1
            segment_out_file = self._get_segment_out_file(parsed_segment, occurence if total_segment_occurence[parsed_segment.get_type()] > 1 else None)
            self._print_to_file([parsed_segment], "", "", segment_out_file)
            files.append(segment_out_file)
        if len(files):
            print("write to files: ", ", ".join(files))

    def _get_segment_out_file(self, parsed_segment, occurence):
        """This method generates a filename using supplied segment and output directory
        """
        occurence_str = ("_" + str(occurence)) if occurence else ""
        segment_out_file = "%s%s.dump" % (parsed_segment.get_name(), occurence_str)
        return os.path.join(self._out_dir, segment_out_file)

    def _print_to_file(self, parsed_segment_db, title, segment_separator, file_path):
        """This method prints the parsed segments to a file.
        """
        with open(file_path, "w") as out_file:
            for notice_section in self._top_notice_db:
                out_file.write(notice_section + "\n")
            if title:
                out_file.write(title + "\n")
            for seg in parsed_segment_db:
                if segment_separator:
                    out_file.write(segment_separator + "\n")
                parsed_seg = seg.get_parsed_data()
                for field in parsed_seg:
                    out_file.write(field + "\n")
            if segment_separator:
                out_file.write(segment_separator + "\n")

    @classmethod
    def _get_fixed_field(cls, field):
        if str(field).find("Warning[") != -1:
            fixed_field = "Warning"
        else:
            fixed_field = field
        return fixed_field

    @classmethod
    def _build_body_msg(cls, field):
        """This method the body string of a line base on the field content.
        """
        if field == "                    Segment":
            body = " - "
        elif field == "RAW DATA":
            body = ":"
        elif str(field).find("Warning[") != -1:
            body = ":"
        elif field.find("DWORD") != cs.PARSER_STRING_NOT_FOUND:
            body = ":"
        else:
            body = " = "
        return body
