# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import os

from .crawler import STECrawler
from .providers.provider import Provider

__all__ = ['STEParser']


class STEParser:
    """A class that represents steering parser.

    :param Provider provider: A provider to fetch STE raw data
    :param io.TextIOWrapper output_file: redirect the output to file object
    :param Dict metadata: Additional metadata for analyzing packet
    """
    def __init__(self, provider, output_file=None, metadata=None):
        self.provider = provider
        self.output_file = output_file
        self.metadata = metadata or {}

    def analyze(self, packet):
        """Analyze the offloaded steering information of a single packet.

        :param Packet packet: The packet to analyze
        """
        print('PACKET_DATA', file=self.output_file)
        print(packet, file=self.output_file, end=os.linesep * 2)
        crawler = STECrawler(self.provider, packet.data, self.metadata)
        crawler.crawl(self.output_file)

    def parse(self, packets):
        """Parse and analyzing the offloaded steering information of packets

        :param list[Packet] packets: A list of packets to parse
        """
        last_index = len(packets)
        for index, packet in enumerate(packets, start=1):
            self.analyze(packet)
            if index < last_index:
                print(file=self.output_file)
