# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import binascii
import os

from .constants import PacketType


__all__ = ['Packet', 'PCAPPacket', 'RawPacket']


class Packet:
    """A class that represents a packet.

    :param bytes data: The raw data of packet
    :param int length: The length of packet
    """
    def __init__(self, data, length=0, *args, **kwargs):
        self.data = data
        self.length = length or len(data)

    def __str__(self):
        data = binascii.b2a_hex(self.data).decode('utf-8')
        hex_str = ' '.join(data[i:i + 2] for i in range(0, len(data), 2))
        lines = (hex_str[i:i + 48].rstrip() for i in range(0, len(hex_str), 48))
        return os.linesep.join(lines).upper()

    def __bytes__(self):
        return self.data

    def __len__(self):
        return self.length


class PCAPPacket(Packet):
    """A class that represents a PCAP packet.

    :param bytes data: The raw data of packet
    :param int length: The length of packet
    :param int timestamp: The timestamp when a packet was captured
    """
    __product_type__ = PacketType.PCAP.value

    def __init__(self, data, length, timestamp):
        super().__init__(data, length)
        self.timestamp = timestamp


class RawPacket(Packet):
    """A class that represents a raw data packet."""
    __product_type__ = PacketType.RAW.value
