# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import argparse
import os

from . import api
from . import dmlib

__all__ = ['HexadecimalIntegerAction', 'MSTDeviceAction',
           'UnsignedIntegerAction', 'WritableFileType']


class HexadecimalIntegerAction(argparse.Action):
    """A class that represents argument parser action for extended integer."""
    def __call__(self, parser, namespace, values, option_string=None):
        # check on which integer's base to use
        base = 16 if values.startswith('0x') else 10
        try:
            # convert numeric string to integer
            value = int(values, base=base)
        except ValueError:
            raise argparse.ArgumentError(
                self, 'invalid int value: \'{}\''.format(values))
        else:
            setattr(namespace, self.dest, value)


class MSTDeviceAction(argparse.Action):
    """A class that represents argument parser action for supported devices."""
    def __call__(self, parser, namespace, values, option_string=None):
        # get device information by the given MST device name argument
        try:
            dev_info = dmlib.get_mst_device_info(values)
        except Exception as err:
            raise argparse.ArgumentError(self, err)
        # check if the given MST device is supported
        if api.is_mst_device_supported(dev_info.hw_dev_id) is False:
            kind = dev_info.type.split()[0]
            err = '{} device ({}) is not supported'.format(kind, dev_info.name)
            raise argparse.ArgumentError(self, err)
        setattr(namespace, self.dest, dev_info.name)


class UnsignedIntegerAction(HexadecimalIntegerAction):
    """A class that represents argument parser action for unsigned integer."""
    def __call__(self, parser, namespace, values, option_string=None):
        super().__call__(parser, namespace, values, option_string)
        # check if the given value is a non-negative number
        value = getattr(namespace, self.dest)
        if value < 0:
            raise argparse.ArgumentError(
                self, 'non-negative number required: {}'.format(value))


class WritableFileType(argparse.FileType):
    """A class that represents writable file type."""
    def __init__(self, *args, **kwargs):
        super().__init__('w', *args, **kwargs)

    def __call__(self, string):
        # makes all intermediate-level directories for the leaf directory.
        os.makedirs(os.path.dirname(string), exist_ok=True)
        return super().__call__(string)
