# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

from functools import singledispatch

from .constants import SUPPORTED_DEVICES
from .dumpers import PacketDumper
from .dmlib import get_mst_device_info
from .loaders import PacketLoader
from .packet import Packet
from .parser import STEParser
from .providers.provider import Provider


def create_packet_dumper(kind, path, *args, **kwargs):
    """Creates a new packet dumper object by a specified type.

    :param str kind: The type of the packet dumper to be created
    :param str path: The output file path for writing packets
    :param args: Arbitrary positional arguments for specific packet dumper
    :param kwargs: Arbitrary keyword arguments for specific packet dumper
    :return: A new packet dumper instance.
    :rtype: PacketDumper
    """
    return PacketDumper.create(kind, path, *args, **kwargs)


def create_packet_loader(kind, path, *args, **kwargs):
    """Creates a new packet loader object by a specified type.

    :param str kind: The type of the packet loader to be created
    :param str path: The input file path which contains the packets
    :param args: Arbitrary positional arguments for specific packet loader
    :param kwargs: Arbitrary keyword arguments for specific packet loader
    :return: A new packet loader instance.
    :rtype: PacketLoader
    """
    return PacketLoader.create(kind, path, *args, **kwargs)


def create_provider(kind, *args, **kwargs):
    """Creates a new provider object by a specified type.

    :param str kind: The type of the provider to be created
    :param args: Arbitrary positional arguments for specific provider
    :param kwargs: Arbitrary keyword arguments for specific provider
    :return: A new provider instance.
    :rtype: Provider
    """
    return Provider.create(kind, *args, **kwargs)


def create_steering_parser(provider, output_file, metadata=None):
    """Creates a new steering parser object.

    :param Provider provider: A provider to fetch STE raw data by ICM address
    :param io.TextIOWrapper output_file: redirect the output to file object
    :param Dict metadata: A specified metadata for analyzing packet
    :return: A new steering parser instance.
    :rtype: STEParser
    """
    return STEParser(provider, output_file, metadata)


def get_packet_dumper_types():
    """Returns a list of all packet dumper types.

    :return: A list of packet dumper types.
    :rtype: list[str]
    """
    return PacketDumper.get_subclasses_types()


def get_packet_loader_types():
    """Returns a list of all packet loader types.

    :return: A list of packet loader types.
    :rtype: list[str]
    """
    return PacketLoader.get_subclasses_types()


def get_provider_types():
    """Returns a list of all providers types.

    :return: A list of providers types.
    :rtype: list[str]
    """
    return Provider.get_subclasses_types()


def dump_packets(dumper_type, path, packets, *args, **kwargs):
    """Dumps a list of packets into an output file.

    :param str dumper_type: The type of the packet dumper to be used
    :param path: The output file path for writing packets
    :param list[Packet] packets: A list of packets to be saved
    :param args: Arbitrary positional arguments for specific packet dumper
    :param kwargs: Arbitrary keyword arguments for specific packet dumper
    """
    dumper = create_packet_dumper(dumper_type, path, *args, **kwargs)
    dumper.dump(packets)


def load_packets(loader_type, path, *args, **kwargs):
    """Loads a list of packets from an input file.

    :param str loader_type: The type of the packet loader to be used
    :param str path: The input file path which contains the packets
    :param args: Arbitrary positional arguments for specific packet loader
    :param kwargs: Arbitrary keyword arguments for specific packet loader
    :return: a list of packets.
    :rtype: list[Packet]
    """
    loader = create_packet_loader(loader_type, path, *args, **kwargs)
    return loader.load()


@singledispatch
def is_mst_device_supported(device_name):
    """Returns whether the given MST device is supported.

    :param str device_name: The name of the MST device
    :return: ``True`` if the MST device is supported otherwise ``False``.
    :rtype: bool
    """
    dev_info = get_mst_device_info(device_name)
    return is_mst_device_supported(dev_info.hw_dev_id)


@is_mst_device_supported.register(int)
def _(hw_dev_id):
    return hw_dev_id in SUPPORTED_DEVICES
