#!/bin/bash

g_remove_pkg_cmd="rpm -e --allmatches --nodeps"
g_kernel_package="kernel-mft"
g_query_pkg="rpm -q"
installed_packages=""
g_help_op="--help"
g_force_op="--force"
g_force=0
g_no_user_op="--no-user"
g_no_user=0
g_no_kernel_op="--no-kernel"
g_no_kernel=0

g_machine_name=`uname -m`

if [[ -x /usr/bin/lsb_release ]]; then
    if [[ `lsb_release -s -i | tr '[:upper:]' '[:lower:]'` == ubuntu ]] || [[ `lsb_release -s -i | tr '[:upper:]' '[:lower:]'` == debian ]]; then
        g_package_type="deb"
    fi
fi
if [[ g_package_type != "deb" ]] && [[ -f /etc/os-release ]]; then
# In some cases lsb_release is not available on Debian machine so we need to check os-release
    if [[ `grep -is "ubuntu\|debian" /etc/os-release` ]]; then
        g_package_type="deb"
    fi
fi
if [[ g_package_type != "deb" ]]; then
    if [[ `cat /etc/issue | head -1` == *Debian* ]] || [[ -f /usr/bin/apt-get ]] || [[ -f /etc/debian_version ]]; then
        g_package_type="deb"
    fi
fi
if [[ ${g_package_type} == "deb" ]]; then
    g_remove_pkg_cmd="dpkg --purge --force-depends"
    g_kernel_package="kernel-mft-dkms"
    g_query_pkg="dpkg -s"
fi

function print_help () {
    cat<<EOF
Usage: mft_uninstall.sh [${g_help_op}] [${g_force_op}]
Options:
    ${g_help_op}          : Print this message.
    ${g_force_op}         : Remove MFT in none interactive mode.
    ${g_no_user_op}       : Do not uninstall user objects.
    ${g_no_kernel_op}     : Do not uninstall kernel objects.

EOF

}

function get_user_options () {
    while [ "$1" ]; do
        case $1 in
            "${g_help_op}")
                print_help
                exit 0
                ;;
            "${g_force_op}")
                g_force=1
                ;;
            "${g_no_user_op}")
                g_no_user=1
                ;;
            "${g_no_kernel_op}")
                g_no_kernel=1
                ;;
            *)
            echo "-E- Bad switch \"$1\""
            print_help
            exit 1
        esac
        shift
    done
}

function ask_before_remove() {
    echo "This program will remove: $1 packages from your machine." 
    read -r -p "Do you want to continue? [y/N] " response
    case $response in
        [yY][eE][sS]|[yY]) 
            # Not doing anything - just continue with the remove flow
            ;;
        *)
            # Exit with RC=1
            exit 1
            ;;
    esac
}

# Only root can install this package.
if [ $UID != 0 ]; then
    echo_error "You must be root to install MFT"
    exit 1
fi

# Get user options.
get_user_options $@

exit_value=0
conf_dir=/etc/mft
rpms_dir=${conf_dir}/RPMS
python_dir=/usr/lib64/mft/python_tools
if [ -d ${rpms_dir} ]; then
    if [ ${g_no_user} != 1 ]; then
        for f in ${rpms_dir}/* ; do
            if [ -f $f ]; then
                pkg_name=`basename $f`
                ${g_query_pkg} ${pkg_name} &> /dev/null
                RC=$?
                if [ "${RC}" == "0" ]; then
                    installed_packages="${pkg_name} ${installed_packages}"
                fi
            fi
        done
    fi
    # Collect the kernel package 
    if [ ${g_no_kernel} != 1 ]; then
        ${g_query_pkg} ${g_kernel_package} &> /dev/null
        RC=$?
        if [ "${RC}" == "0" ]; then
            installed_packages="${installed_packages} ${g_kernel_package}"
        fi
    fi
    if [[ ${installed_packages} == "" ]]; then
        echo "Nothing to do."
    else
        if [ $g_force == 0 ]; then
            ask_before_remove "${installed_packages}"
        fi
        echo -e "-I- Removing all installed mft packages: ${installed_packages}"
        for pkg in ${installed_packages}; do
            ${g_remove_pkg_cmd} ${pkg} &> /dev/null
            rc=$?
            if test $rc -ne 0
            then
                echo -e "-E- Failed to remove package: ${g_remove_pkg_cmd} ${pkg} - Failed.\nTry to remove it manually !"
                exit_value=1
            fi
        done  
    fi
    if [ "${exit_value}" == "0" ]; then
        if [ -d ${conf_dir} ]; then
            rm -rf ${conf_dir}
        fi
        if [ -d ${python_dir} ]; then
            rm -rf ${python_dir}
        fi
    fi 
else
    echo "-E- Failed to detect installed RPMS, No such directory ${rpms_dir}"
    exit_value=1
fi
exit ${exit_value}
