/*
 * Copyright (c) 2021-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * Copyright (c) 2004-2009 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/*
 * Abstract:
 * 	Declaration of osm_inform_rec_t.
 *	This object represents an IBA Inform Record.
 *	This object is part of the OpenSM family of objects.
 *
 * Author:
 *    Eitan Zahavi, Mellanox
 */

#ifndef _OSM_INFR_H_
#define _OSM_INFR_H_

#include <iba/ib_types.h>
#include <complib/cl_qmap.h>
#include <complib/cl_spinlock.h>
#include <opensm/osm_subnet.h>
#include <opensm/osm_madw.h>
#include <opensm/osm_log.h>
#include <opensm/osm_sa.h>

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else				/* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif				/* __cplusplus */

BEGIN_C_DECLS
/****h* OpenSM/Inform Record
* NAME
*	Inform Record
*
* DESCRIPTION
*	The Inform record encapsulates the information needed by the
*	SA to manage InformInfo registrations and sending Reports(Notice)
*	when SM receives Traps for registered LIDs.
*
*	The inform records is not thread safe, thus callers must provide
*	serialization.
*
*	This object should be treated as opaque and should be
*	manipulated only through the provided functions.
*
* AUTHOR
*    Eitan Zahavi, Mellanox
*
*********/
/****s* OpenSM: Inform Record/osm_infr_t
* NAME
*	osm_infr_t
*
* DESCRIPTION
*	Inform Record structure.
*
*	The osm_infr_t object should be treated as opaque and should
*	be manipulated only through the provided functions.
*
* SYNOPSIS
*/
typedef struct osm_infr {
	cl_list_item_t list_item;
	osm_bind_handle_t h_bind;
	osm_sa_t *sa;
	osm_mad_addr_t report_addr;
	ib_inform_info_record_t inform_record;
	uint8_t is_trusted;
} osm_infr_t;
/*
* FIELDS
*	list_item
*		List Item for qlist linkage.  Must be first element!!
*
*	h_bind
*		A handle of lower level mad srvc
*
*	sa
*		A pointer to osm_sa object
*
*	report_addr
*		Report address
*
*	inform_record
*		The Inform Info Record
*
* 	is_trusted
* 		Indicator that InformInfo subscription request was generated by trusted source
*
* SEE ALSO
*********/

/****f* OpenSM: Inform Record/osm_infr_new
* NAME
*	osm_infr_new
*
* DESCRIPTION
*	Allocates and initializes a Inform Record for use.
*
* SYNOPSIS
*/
osm_infr_t *osm_infr_new(IN const osm_infr_t * p_infr_rec);
/*
* PARAMETERS
*	p_inf_rec
*		[in] Pointer to IB Inform Record
*
* RETURN VALUES
*	pointer to osm_infr_t structure.
*
* NOTES
*	Allows calling other inform record methods.
*
* SEE ALSO
*	Inform Record, osm_infr_delete
*********/

/****f* OpenSM: Inform Record/osm_infr_delete
* NAME
*	osm_infr_delete
*
* DESCRIPTION
*	Destroys and deallocates the osm_infr_t structure.
*
* SYNOPSIS
*/
void osm_infr_delete(IN osm_infr_t * p_infr);
/*
* PARAMETERS
*	p_infr
*		[in] Pointer to osm_infr_t structure
*
* SEE ALSO
*	Inform Record, osm_infr_new
*********/

/****f* OpenSM: Inform Record/osm_infr_get_by_rec
* NAME
*	osm_infr_get_by_rec
*
* DESCRIPTION
*	Find a matching osm_infr_t in the subnet DB by inform_info_record
*
* SYNOPSIS
*/
osm_infr_t *osm_infr_get_by_rec(IN osm_subn_t const *p_subn,
				IN osm_log_t * p_log,
				IN osm_infr_t * p_infr_rec);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	p_log
*		[in] Pointer to the log object
*
*	p_inf_rec
*		[in] Pointer to an inform_info record
*
* RETURN
*	The matching osm_infr_t
* SEE ALSO
*	Inform Record, osm_infr_new, osm_infr_delete
*********/

void osm_infr_insert_to_db(IN osm_subn_t * p_subn, IN osm_log_t * p_log,
			   IN osm_infr_t * p_infr);

void osm_infr_remove_from_db(IN osm_subn_t * p_subn, IN osm_log_t * p_log,
			     IN osm_infr_t * p_infr);

ib_api_status_t osm_send_report(IN osm_subn_t * p_subn,
				IN ib_mad_notice_attr_t * p_ntc);

int osm_infr_match_notice_to_db(IN osm_subn_t * p_subn, IN osm_log_t * p_log,
				IN osm_mad_addr_t * p_mad_addr,
				IN ib_mad_notice_attr_t * p_ntci,
				IN int remove);

/****f* OpenSM: Inform Record/osm_infr_remove_by_guids
* NAME
*	osm_infr_remove_by_guids
*
* DESCRIPTION
*	Remove and delete from Database all InformInfo Records of specified
*	port GUIDs.
*
* SYNOPSIS
*/
void osm_infr_remove_by_guids(IN osm_subn_t * p_subn, IN osm_log_t * p_log,
			      IN ib_net64_t * guids, IN int num_guids);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to Subnet structure
*
*	p_log
*		[in] Pointer to osm_log_t
*
*	guids
*		[in] Array of GUIDs
*
*	num_guids
*		[in] GUIDs array length
*
*
* RETURN VALUES
*	This function does not return a value.
*
* SEE ALSO
*	Service Record, osm_infr_insert_to_db, osm_infr_delete
*********/

/****f* OpenSM: Inform Record/osm_report_notice
* NAME
*	osm_report_notice
*
* DESCRIPTION
* Once a Trap was received by the osm_trap_rcv, or a Trap sourced in
* the SM was sent (Traps 64-67) this routine is called with a copy of
* the notice data.
* Given a notice attribute - compare and see if it matches the InformInfo
* Element and if it does - call the Report(Notice) for the
* target QP registered by the address stored in the InformInfo element
*
* SYNOPSIS
*/
ib_api_status_t osm_report_notice(IN osm_log_t * p_log, IN osm_subn_t * p_subn,
				  IN ib_mad_notice_attr_t * p_ntc,
				  IN uint64_t tid, IN uint64_t trap_tid);
/*
* PARAMETERS
*	p_rcv
*		[in] Pointer to the trap receiver
*
*	p_subn
*		[in] Pointer to the subnet object
*
*	p_ntc
*		[in] Pointer to a copy of the incoming trap notice attribute.
*
*	tid
*		[in] Transaction ID
*
*	trap_tid
*		[in] Transaction ID in Trap
*
* RETURN
*	IB_SUCCESS on good completion
*
* SEE ALSO
*	Inform Record, osm_trap_rcv
*********/


/****f* OpenSM: Inform Record/osm_report_notice_64_67
* NAME
*	osm_report_notice_64_67
*
* DESCRIPTION
* 	This routine creates a notice object for traps 64 - 67, and invokes
* 	osm_report_notice with the created object.
* 	The purpose of this routine is to generate and submit GID in/out and
* 	MGID in/out notices.
*
* SYNOPSIS
*/

ib_api_status_t osm_report_notice_64_67(osm_subn_t * p_subn, ib_gid_t * p_gid,
					uint16_t trap_num);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	p_gid
*		[in] Pointer to a GID for the notice
*
*	trap_num
*		[in] Trap number, must be between 64 and 67.
*
* RETURN
*	IB_SUCCESS on good completion
*
* SEE ALSO
*	Inform Record, osm_report_notice
*********/

/****f* OpenSM: Inform Record/osm_report_vs_notice
* NAME
*	osm_report_vs_notice
*
* DESCRIPTION
* 	This routine creates a notice object for VS traps generated by SM, and
* 	invokes	osm_report_notice with the created object.
* 	The purpose of this routine is to generate and submit SA security notices.
*
* SYNOPSIS
*/

ib_api_status_t osm_report_vs_notice(osm_subn_t *p_subn,
				     struct _vendor_data_details *notice_data,
				     uint8_t type);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	notice_data
*		[in] Pointer to the data needed to fill the notice
*
*	type
*		[in] Notice type
*
* RETURN
*	IB_SUCCESS on good completion
*
* SEE ALSO
*	Inform Record, osm_report_notice
*********/

/****f* OpenSM: Inform Record/osm_report_sa_security_notice
* NAME
*	osm_report_sa_security_notice
*
* DESCRIPTION
* 	This routine creates a notice object for traps 1300 - 1305, and invokes
* 	osm_report_notice with the created object.
* 	The purpose of this routine is to generate and submit SA security notices.
*
* SYNOPSIS
*/

ib_api_status_t osm_report_sa_security_notice(osm_subn_t *p_subn,
					      struct _vendor_data_details *notice_data);

/****f* OpenSM: Inform Record/osm_report_notices_1303_1305
* NAME
*	osm_report_notices_1303_1305
*
* DESCRIPTION
* 	This routine fills notice data for traps 1303 - 1305, and invokes
* 	osm_report_vs_notice with the created object.
*
* SYNOPSIS
*/
void osm_report_notices_1303_1305(osm_subn_t *p_subn, osm_alias_guid_t *p_req_aguid,
				  osm_alias_guid_t *p_target_aguid, uint16_t trap_num);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	p_req_aguid
*		[in] Pointer to the sender alias guid
*
*	p_target_aguid
*		[in] Pointer to the offender alias guid
*
*	trap_num
*		[in] Trap number in network order
*
* SEE ALSO
*	osm_report_vs_notice
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1306_1307
* NAME
*	osm_report_notice_1306_1307
*
* DESCRIPTION
* 	This routine creates a notice object for traps 1306-1307,
* 	and invokes osm_report_notice with the created object.
*
* SYNOPSIS
*/
void osm_report_notice_1306_1307(osm_subn_t *p_subn, ib_net16_t lid,
				 ib_net64_t unauthorized_sm_guid,
				 boolean_t is_dr_notice, osm_dr_path_t *p_dr_path,
				 ib_net16_t trap_num);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	lid
*		[in] The lid of the unauthorized SM or 0xFF.
*
*	unauthorized_sm_guid
*		[in] A guid of an SM that was found on a port that
*		     is not part of the allowed SM guids.
*
*	is_dr_notice
*		[in] Indication weather this Notice results from Direct route SMP
*
*	p_dr_path
*		[in] In case the is_dr_notice=True the direct route to
*		     the unauthorized port that caused the notice
*
*	trap_num
*		[in] Trap number in network order
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1309
* NAME
*	osm_report_notice_1309
*
* DESCRIPTION
* 	This routine creates a notice object for trap 1309,
* 	and invokes osm_report_notice with the created object.
*
* SYNOPSIS
*/
void osm_report_notice_1309(osm_subn_t *p_subn, ib_net16_t lid,
			    ib_net64_t physp_guid, const ib_smp_t *p_smp,
			    ib_net64_t sm_key);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	lid
*		[in] The lid of the discovered SM
*
*	physp_guid
*		[in] The guid of the discovered SM
*
*	p_smp
*		[in] A pointer to the smp struct
*
*	sm_key
*		[in] The sm_key of the discovered SM
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1310
* NAME
*	osm_report_notice_1310
*
* DESCRIPTION
* 	This routine creates a notice object for trap 1310 in order to report
* 	duplicated Node GUID, and invokes osm_report_notice with the created
* 	object.
*
* SYNOPSIS
*/
void osm_report_notice_1310(osm_subn_t *p_subn,
			    ib_net64_t node_guid,
			    ib_net64_t peer_node_guid_1, uint8_t peer_port_1,
			    ib_net64_t peer_node_guid_2, uint8_t peer_port_2);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	node_guid
*		[in] Duplicated Node GUID to report
*
*	peer_node_guid_1
*		[in] Node GUID of one device connected to node with duplicated
*		GUID
*
*	peer_port_1
*		[in] Port number of one device connected to node with duplicated
*		GUID
*
*	peer_node_guid_2
*		[in] Node GUID of one device connected to node with duplicated
*		GUID
*
*	peer_port_2
*		[in] Port number of one device connected to node with duplicated
*		GUID
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1311
* NAME
*	osm_report_notice_1311
*
* DESCRIPTION
* 	This routine creates a notice object for trap 1311 in order to report
* 	duplicated Port GUID, and invokes osm_report_notice with the created
* 	object.
*
* SYNOPSIS
*/
void osm_report_notice_1311(osm_subn_t *p_subn,
			    ib_net64_t port_guid,
			    ib_net64_t node_guid_1, uint8_t port_1, uint16_t index_1,
			    ib_net64_t node_guid_2, uint8_t port_2, uint16_t index_2);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	port_guid
*		[in] Duplicate Port GUID to report
*
*	node_guid_1
*		[in] The Node GUID of conflicting port 1
*
*	port_1
*		[in] The port number conflicting port 1
*
*	index_1
*		[in] Virtual port index of conflicting port 1
*
*	node_guid_2
*
*		[in] The Node GUID of conflicting port 2
*
*	port_2
*		[in] The port number conflicting port 2
*
*	index_2
*		[in] Virtual port index of conflicting port 2
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1312
* NAME
*	osm_report_notice_1312
*
* DESCRIPTION
* 	This routine creates a notice object for trap 1312 in order to report
* 	node reboot, and invokes osm_report_notice with the created
* 	object.
*
* SYNOPSIS
*/
void osm_report_notice_1312(osm_subn_t *p_subn,
			    ib_net64_t port_guid,
			    ib_net64_t node_guid,
			    uint8_t port);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	port_guid
*		[in] The Port GUID of the rebooted Port
*
*	node_guid
*		[in] The Node GUID of rebooted NODE
*
*	port
*		[in] The port number of the rebooted Port
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1313
* NAME
*	osm_report_notice_1313
*
* DESCRIPTION
* 	This routine creates a notice object for trap 1313 in order to report
* 	router LID assignment error.
*
* SYNOPSIS
*/
void osm_report_notice_1313(osm_subn_t *p_subn, osm_vs_trap_lid_router_error_syndrom_t syndrome);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*	syndrome
*		[in] syndrome that tells a reason for the trap
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1315
* NAME
*	osm_report_notice_1315
*
* DESCRIPTION
* 	This routine creates a notice object for trap 1315 in order to report
* 	topoconfig file error.
*
* SYNOPSIS
*/
void osm_report_notice_1315(osm_subn_t *p_subn, osm_vs_trap_topoconfig_parsing_error_syndrom_t error_type);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	error_type
*		[in] the type of the error that caused the trap
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1316
* NAME
*	osm_report_notice_1316
*
* DESCRIPTION
* 	This routine creates a notice object for trap 1316 in order to report
* 	link inconsistency between what was found in the subnet and what was defined in the topoconfig file.
*
* SYNOPSIS
*/
void osm_report_notice_1316(osm_subn_t *p_subn,
			    ib_net64_t node_guid, uint8_t port,
			    ib_net64_t neighbor_node_guid, uint8_t neighbor_port,
			    uint8_t neighbor_node_type,
			    ib_net64_t expected_neighbor_node_guid, uint8_t expected_neighbor_port,
			    uint8_t expected_neighbor_node_type);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*
*	node_guid
*		[in] The Node GUID of the device
*
*	port
*		[in] The port number
*
*	neighbor_node_guid
*		[in] The neighbor node GUID that was discovered
*
*	neighbor_port
*		[in] The neighbor port number that was discovered
*
*	expected_neighbor_node_guid
*		[in] The topoconfig expected neighbor node GUID
*
*	expected_neighbor_port
*		[in] The topoconfig expected neighbor port
*
*	expected_neighbor_node_type
*		[in] The topoconfig expected neighbor node type
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1601
* NAME
*	osm_report_notice_1601
*
* DESCRIPTION
*	This routine creates a notice object for trap 1601 in order to report
*	virtual port access violation.
*
* SYNOPSIS
*/
void osm_report_notice_1601(osm_subn_t *p_subn,
			    uint64_t virtual_guid,
			    uint64_t physical_guid,
			    uint64_t virtual_group_id,
			    uint64_t physical_group_id,
			    uint16_t virtual_index,
			    uint8_t tenant_notice_type);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*	virtual_guid
*		[in] The guid of the virtual port.
*	physical_guid
*		[in] The guid of the physical port.
*	virtual_group_id
*		[in] The map group id of the virtual port.
*	physical_group_id
*		[in] The map group id of the physical port.
*	virtual_index
*		[in] Index of the virtual port on the physical port.
*	tenant_notice_type
*		[in] Type of tenant violation
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1605
* NAME
*      osm_report_notice_1605
*
* DESCRIPTION
*      This routine creates a notice object for trap 1605 in order to report
*      routing engine action of a switch removal.
*
* SYNOPSIS
*/
void osm_report_notice_1605(osm_subn_t *p_subn, uint64_t guid);
/*
* PARAMETERS
*      p_subn
*              [in] Pointer to the subnet object
*      guid
*              [in] Switch guid.
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1606
* NAME
*      osm_report_notice_1606
*
* DESCRIPTION
*      This routine creates a notice object for trap 1606 in order to report
*      routing engine action of a switch recovery.
*
* SYNOPSIS
*/
void osm_report_notice_1606(osm_subn_t *p_subn, uint64_t guid);
/*
* PARAMETERS
*      p_subn
*              [in] Pointer to the subnet object
*      guid
*              [in] Switch guid.
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1609
* NAME
*	osm_report_notice_1609
*
* DESCRIPTION
*	This routine creates a notice object for trap 1609 in order to report
*	an ISSU error condition.
*
* SYNOPSIS
*/
void osm_report_notice_1609(osm_subn_t *p_subn, uint64_t guid, uint8_t state);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*	guid
*		[in] The guid of the switch.
*	state
*		[in] ISSU state for the given guid.
*
*********/

/****f* OpenSM: Inform Record/osm_report_notice_1610
* NAME
*	osm_report_notice_1610
*
* DESCRIPTION
*	This routine creates a notice object for trap 1610 in order to report
*	an ISSU recovery condition.
*
* SYNOPSIS
*/
void osm_report_notice_1610(osm_subn_t *p_subn, uint64_t guid, uint8_t state);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to the subnet object
*	guid
*		[in] The guid of the switch.
*	state
*		[in] ISSU state for the given guid.
*
*********/

END_C_DECLS
#endif				/* _OSM_INFR_H_ */
